program APPROXLOCATION;
{--------------------------------------------------------------------}
{  Alg2'4.pas   Pascal program for implementing Algorithm 2.4        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.4 (Approximate Location of Roots).                    }
{  Section   2.3, Initial Approximations and Convergence Criteria,   }
{  Page 70                                                           }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Epsilon = 0.05;
    MN = 201;
    FunMax = 9;

  type
    VECTOR = array[0..MN] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, Working);

  var
    FunType, Inum, N, M, Sub: integer;
    A, B, H, Rnum: real;
    X, R, Y: VECTOR;
    Stat, State: STATUS;
    Ans: CHAR;
    Mess: LETTERS;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := X * X - EXP(X);
      2: 
        F := X * X * X - X * X - X - 1;
      3: 
        F := X - COS(X);
      4: 
        F := SIN(X) - 2 * COS(2 * X);
      5: 
        F := COS(X) + 1 / (1 + X * X);
      6: 
        F := (X - 2) * (X - 2) - LN(X);
      7: 
        F := 2 * X - SIN(X) / COS(X);
      8: 
        F := X * X * X - X - 3;
      9: 
        F := 4 * X * X * X - 2 * X - 6;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1: 
        WRITELN('F(X) = X*X - EXP(X)');
      2: 
        WRITELN('F(X) = X*X*X -  X*X - X - 1');
      3: 
        WRITELN('F(X) = X - COS(X)');
      4: 
        WRITELN('F(X) = SIN(X) - 2*COS(2*X)');
      5: 
        WRITELN('F(X) = COS(X) + 1/(1 + X*X)');
      6: 
        WRITELN('F(X) = (X-2)^2 - LN(X)');
      7: 
        WRITELN('F(X) = 2*X - TAN(X)');
      8: 
        WRITELN('F(X) = X*X*X - X - 3');
      9: 
        WRITELN('F(X) = 4*X*X*X - 2*X - 6');
      else
        WRITE('F(X) = X*X*X - 3*X - 2');
    end;
  end;

  procedure LOCATE (A, B: real; N: integer; Epsilon: real; var R: VECTOR; var M: integer);
    var
      K: integer;
      H, MaxY, MinY, Slope, Small: real;
  begin
    H := (B - A) / N;
    for K := 0 to N do
      begin
        X[K] := A + H * K;
        Y[K] := F(X[K]);
      end;
    MaxY := Y[0];
    for K := 1 to N do
      if MaxY < Y[K] then
        MaxY := Y[K];
    MinY := Y[0];
    for K := 1 to N do
      if MinY > Y[K] then
        MinY := Y[K];
    Small := (MaxY - MinY) * Epsilon;
    M := 0;
    Y[N + 1] := Y[N];
    for K := 1 to N do
      begin
        if (Y[K - 1] * Y[K]) <= 0 then
          begin
            M := M + 1;
            R[M] := (X[K - 1] + X[K]) / 2;
          end;
        Slope := (Y[K] - Y[K - 1]) * (Y[K + 1] - Y[K]);
        if (ABS(Y[K]) < Small) and (Slope < 0) then
          begin
            M := M + 1;
            R[M] := X[K];
          end;
      end;
  end;

  procedure MESSAGE (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('                   APPROXIMATE LOCATION OF ROOTS');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('     Approximate location(s) for the zero(s) of the function F(X) will');
    WRITELN;
    WRITELN;
    WRITELN('be found in [A,B]. The interval will be subdivided into N subintervals');
    WRITELN;
    WRITELN;
    WRITELN('{[X   ,X ]: k=1,2,...,N}, and a search is performed to determine if F(X) ');
    WRITELN('   k-1  k');
    WRITELN;
    WRITELN('changes sign or if F(X) is close to zero on each subinterval.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('                   Press the <ENTER> key.  ');
    READLN(Ans);
    CLRSCR;
    WRITELN('             Function choices are:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('     <', K : 2, ' >   ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '     SELECT your function  < 1 - 9 > ?  ';
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure GETPOINTS (var A, B: real; var N: integer);
    var
      T: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('     You chose to find the approximate locations for the zeros of zero of:');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('in an interval [A,B].  The interval must be subdivided into N subintervals');
    WRITELN;
    WRITELN;
    WRITELN('{[X   ,X ]: k=1,2,...,N}.  Then a search is performed to determine if F(X) ');
    WRITELN('   k-1  k');
    WRITELN;
    WRITELN('changes sign or if  F(X) is close to zero on each subinterval.');
    WRITELN;
    WRITELN;
    WRITELN('     Now the interval [A,B] and integer N must be selected.');
    WRITELN;
    WRITELN;
    Mess := '     ENTER  the  left   endpoint  A = ';
    A := 0;
    WRITE(Mess);
    READLN(A);
    Mess := '     ENTER  the  right  endpoint  B = ';
    B := 1;
    WRITE(Mess);
    READLN(B);
    Mess := '     ENTER number of subintervals N = ';
    N := 91;
    WRITE(Mess);
    READLN(N);
    WRITELN;
    if B < A then
      begin
        T := A;
        A := B;
        B := T;
      end;
  end;                                     {End of PROCEDURE GETPOINTS}

  procedure RESULTS (A, B: real; N: integer; R: VECTOR; M: integer);
    var
      K: integer;
  begin
    H := (B - A) / N;
    CLRSCR;
    WRITELN('A search was performed to find the approximate locations for the zeros of');
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('in the interval  [', A : 15 : 7, ' ,', B : 15 : 7, '  ].');
    WRITELN;
    WRITELN('The interval was subdivided into ', N, ' subintervals');
    WRITELN;
    WRITELN('of equal width   H =', H : 15 : 7);
    WRITELN;
    if M > 0 then
      begin
        if M = 1 then
          WRITELN('     We found  1  approximate location.')
        else
          WRITELN('     We found ', M : 1, ' approximate locations.');
        WRITELN;
        if M = 1 then
          WRITELN('It`s abscissa X(K) and ordinate Y(K) is:')
        else
          WRITELN('Their abscissas X(K) and ordinates Y(K) are:');
        WRITELN;
        for K := 1 to M do
          begin
            WRITELN('X(', K : 3, ') =', R[K] : 15 : 7, '           Y(', K : 3, ') =', F(R[K]) : 15 : 7);
            WRITELN;
            if K mod 5 = 0 then
              begin
                WRITE('                  Press the <ENTER> key.  ');
                READLN(Ans);
                WRITELN;
              end;
          end;
        WRITELN;
      end
    else
      begin
        WRITELN('NO approximate locations for zeros of F(X) were found.');
        WRITELN;
        WRITELN('You could try another search using a smaller step H,');
        WRITELN;
        WRITELN('or you could try a search in a different interval [A,B].');
      end;
  end;

begin                                            {Begin Main Program}
  FunType := 1;
  A := 0;
  B := 1;
  Stat := Working;
  while (Stat = Working) do
    begin
      MESSAGE(FunType);
      State := Computing;
      while (State = Computing) do
        begin
          GETPOINTS(A, B, N);
          LOCATE(A, B, N, Epsilon, R, M);
          RESULTS(A, B, N, R, M);
          WRITELN;
          WRITE('Want to try a different interval [A,B] ? <Y/N>  ');
          READLN(Ans);
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to try a different function  F(X) ? <Y/N>  ');
      READLN(Ans);
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

